/*
 *  Copyright (C) 2004 Mathias Andre <mathias@openbrookes.org>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 */


#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <string.h>
#include <stdio.h>
#include <time.h>

#include <gdk/gdkkeysyms.h>
#include <gtk/gtk.h>
#include <glib.h>
#include <glade/glade.h>

#include "callbacks.h"
#include "interface.h"
#include "../nb_global.h"

#define _(text) gettext(text)

void
create_mainwindow (void)
{
	extern GladeXML * xml;
	GtkWidget * mainwindow;

	/* load the interface */
	xml = glade_xml_new (g_build_path ("/", GLADE_DIR, "main.glade", NULL), NULL, NULL);

	create_tree ();

	/* connect the signals in the interface */
	glade_xml_signal_autoconnect(xml);

	/* blank fields to set the date field */
	blank_fields ();

	create_textview ();

	/* resize the window and vpan to last values  */
	load_interface_positions ();
	
	/* set drag and drap on mainwindow */
	mainwindow = glade_xml_get_widget (xml, "mainwindow");
	gtk_drag_dest_set (mainwindow, GTK_DEST_DEFAULT_ALL, dragdroptarget, 3, GDK_ACTION_COPY|GDK_ACTION_MOVE|GDK_ACTION_LINK);
	g_signal_connect (mainwindow, "drag_data_received", G_CALLBACK (on_drag_received), NULL);
}

void 
create_tree (void)
{
	extern GladeXML * xml;
	extern global * g;
	GtkWidget * tree;
	GtkTreeSelection *selection;
	GtkTreeViewColumn *column;
	GtkCellRenderer *renderer;

	tree = glade_xml_get_widget(xml, "treeview");
	gtk_tree_view_set_model (GTK_TREE_VIEW (tree), GTK_TREE_MODEL (g->store));

	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new_with_attributes ((gchar *) _("Title"), renderer,
							   "text", TITLE_COLUMN,
							   NULL);
	gtk_tree_view_column_set_resizable (column, TRUE);
	gtk_tree_view_column_set_sort_column_id (column, TITLE_COLUMN);
	gtk_tree_view_append_column (GTK_TREE_VIEW (tree), column);

	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new_with_attributes ((gchar *) _("Type"), renderer,
							   "text", TYPE_COLUMN,
							   NULL);
	gtk_tree_view_column_set_resizable (column, TRUE);
	gtk_tree_view_column_set_sort_column_id (column, TYPE_COLUMN);
	gtk_tree_view_append_column (GTK_TREE_VIEW (tree), column);

	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new_with_attributes ((gchar *) _("Status"), renderer,
							   "text", STATUS_COLUMN,
							   NULL);
	gtk_tree_view_column_set_resizable (column, TRUE);
	gtk_tree_view_column_set_sort_column_id (column, STATUS_COLUMN);
	gtk_tree_view_append_column (GTK_TREE_VIEW (tree), column);

	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new_with_attributes ((gchar *) _("Date"), renderer,
							   "text", DATE_COLUMN,
							   NULL);
	gtk_tree_view_column_set_resizable (column, TRUE);
	gtk_tree_view_column_set_sort_indicator (column, TRUE);
	gtk_tree_view_column_set_sort_column_id (column, DATE_COLUMN);
	gtk_tree_view_append_column (GTK_TREE_VIEW (tree), column);

	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new_with_attributes ((gchar *) _("Last update"), renderer,
							   "text", LAST_UPDATE_COLUMN,
							   NULL);
	gtk_tree_view_column_set_resizable (column, TRUE);
	gtk_tree_view_column_set_sort_column_id (column, LAST_UPDATE_COLUMN);
	gtk_tree_view_append_column (GTK_TREE_VIEW (tree), column);

	selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (tree));
	gtk_tree_selection_set_mode (selection, GTK_SELECTION_MULTIPLE);
	g_signal_connect (G_OBJECT (selection), "changed", G_CALLBACK (tree_selection_changed), NULL);
}

void
create_textview (void)
{
	/* create the links tag for the text view + set wrap mode */
	extern GladeXML * xml;
	GtkTextTag * tag;
	GtkWidget * textview = glade_xml_get_widget (xml, "textview");
	GtkTextBuffer * buffer = gtk_text_view_get_buffer ( GTK_TEXT_VIEW (textview) );
	format_textview * format = g_malloc (sizeof (format_textview));	

	/* create the mouse cursors */
	format->hand_cursor = gdk_cursor_new (GDK_HAND2);
	format->text_cursor = gdk_cursor_new (GDK_XTERM);
	format->current_cursor = GDK_XTERM;

	g_object_set_data ( G_OBJECT (textview), "format", format);

	gtk_text_view_set_wrap_mode ( GTK_TEXT_VIEW (textview), GTK_WRAP_WORD_CHAR);

	tag = gtk_text_buffer_create_tag (buffer, "links", "foreground", "blue", NULL);

	g_signal_connect (G_OBJECT (textview), "event-after", G_CALLBACK (on_link_activate), NULL);
	g_signal_connect (G_OBJECT (textview), "motion-notify-event", G_CALLBACK (motion_notify_event), NULL);
	g_signal_connect (G_OBJECT (textview), "visibility-notify-event", G_CALLBACK (visibility_notify_event), NULL);
	g_signal_connect (G_OBJECT (buffer), "changed", G_CALLBACK (scan_url_tag_from_textview), NULL);
}


void
blank_fields (void)
{
	extern GladeXML * xml;
	GtkWidget * entrysubject = glade_xml_get_widget (xml, "entrysubject");
	GtkWidget * textview = glade_xml_get_widget (xml, "textview");
	GtkWidget * optionmenutype = glade_xml_get_widget (xml, "optionmenutype");
	GtkWidget * optionmenustatus = glade_xml_get_widget (xml, "optionmenustatus");
	GtkWidget * note1 = glade_xml_get_widget (xml, "note1");
	GtkWidget * none1 = glade_xml_get_widget (xml, "none1");
	GtkWidget * date = glade_xml_get_widget (xml, "labeldatevalue");
	GtkWidget * treeview = glade_xml_get_widget(xml, "treeview");
	GtkTreeSelection * selection;

	GtkTextBuffer * buffer = gtk_text_view_get_buffer ( GTK_TEXT_VIEW (textview) );
	gtk_text_buffer_set_text (buffer, "", 0);
	gtk_text_view_set_buffer ( GTK_TEXT_VIEW (textview) , buffer);
	gtk_entry_set_text ( GTK_ENTRY (entrysubject) , "");
	gtk_option_menu_set_menu ( GTK_OPTION_MENU (optionmenutype) , note1);
	gtk_option_menu_set_menu ( GTK_OPTION_MENU (optionmenustatus) , none1);
	gtk_label_set_text ( GTK_LABEL (date) , get_current_date ());

	selection = gtk_tree_view_get_selection ( GTK_TREE_VIEW (treeview));
	g_signal_handlers_block_by_func (G_OBJECT (selection), tree_selection_changed, NULL);
	gtk_tree_selection_unselect_all (selection);
	g_signal_handlers_unblock_by_func (G_OBJECT (selection), tree_selection_changed, NULL);
}

void
clear_tree_view (void)
{
	extern global * g;

	gtk_list_store_clear (g->store);
}

gboolean
is_empty_note (void)
{
	extern GladeXML * xml;
	GtkTextIter start;
	GtkTextIter end;

	GtkWidget * entrysubject = glade_xml_get_widget (xml, "entrysubject");
	GtkWidget * textview = glade_xml_get_widget (xml, "textview");

	GtkTextBuffer * buffer = gtk_text_view_get_buffer ( GTK_TEXT_VIEW (textview) );
	gtk_text_buffer_get_start_iter(buffer,&start);
	gtk_text_buffer_get_end_iter(buffer,&end);

	if ( (strlen (gtk_entry_get_text ( GTK_ENTRY (entrysubject) )) == 0) &&
	     (strlen (gtk_text_buffer_get_text(buffer,&start, &end,TRUE)) == 0) )
		return (TRUE);
	else
		return (FALSE);
}

void 
set_sort_func (GtkListStore * store)
{
	GtkTreeSortable * sortable;

	sortable = GTK_TREE_SORTABLE (store);

	gtk_tree_sortable_set_sort_func (sortable, TITLE_COLUMN, sort_iter_compare_string, GINT_TO_POINTER (TITLE_COLUMN), NULL);
	gtk_tree_sortable_set_sort_func (sortable, TYPE_COLUMN, sort_iter_compare_string, GINT_TO_POINTER (TYPE_COLUMN), NULL);
	gtk_tree_sortable_set_sort_func (sortable, STATUS_COLUMN, sort_iter_compare_string, GINT_TO_POINTER (STATUS_COLUMN), NULL);
	gtk_tree_sortable_set_sort_func (sortable, DATE_COLUMN, sort_iter_compare_date, GINT_TO_POINTER (DATE_COLUMN), NULL);
	gtk_tree_sortable_set_sort_func (sortable, LAST_UPDATE_COLUMN, sort_iter_compare_date, GINT_TO_POINTER (LAST_UPDATE_COLUMN), NULL);

}

void
add_note_to_tree (gchar * title, gchar * type, gchar * status, gchar * date, gchar * last_update, note * n, gboolean scroll)
{
	extern global * g;
	extern GladeXML * xml;
	GtkTreeIter   iter;
	gsize bytes_written;
	GtkTreeSelection * selection;
	GtkWidget * treeview = glade_xml_get_widget (xml, "treeview");

	gtk_list_store_append (GTK_LIST_STORE (g->store) , &iter);  /* Acquire an iterator */

	/* convert the string to utf8 and insert them */
	gtk_list_store_set (GTK_LIST_STORE (g->store) , &iter,
			TITLE_COLUMN, g_locale_to_utf8 (title, -1, NULL, &bytes_written, NULL),
			TYPE_COLUMN, g_locale_to_utf8 (type, -1, NULL, &bytes_written, NULL),
			STATUS_COLUMN, g_locale_to_utf8 (status, -1, NULL, &bytes_written, NULL),
			DATE_COLUMN, g_locale_to_utf8 (date, -1, NULL, &bytes_written, NULL),
			LAST_UPDATE_COLUMN, g_locale_to_utf8 (last_update, -1, NULL, &bytes_written, NULL),
			NOTE_POINTER_COLUMN, n,
			-1);

	/* block the selection changed handler and set the selection to added note */
	selection = gtk_tree_view_get_selection ( GTK_TREE_VIEW (treeview));
	g_signal_handlers_block_by_func ( G_OBJECT (selection), tree_selection_changed, NULL);
	gtk_tree_selection_unselect_all (selection);
	gtk_tree_selection_select_iter (selection, &iter);
	g_signal_handlers_unblock_by_func ( G_OBJECT (selection), tree_selection_changed, NULL);

	if ( scroll )
	{
		/* scroll the added note */
		gtk_tree_view_scroll_to_cell ( GTK_TREE_VIEW (treeview),
				gtk_tree_model_get_path ( gtk_tree_view_get_model (GTK_TREE_VIEW (treeview)), &iter),
				NULL,
				FALSE,
				0.0,
				0.0);
	}
}

void
update_tree_with_current (gchar * title, gchar * type, gchar * status, gchar * last_update)
{
	extern global * g;
	extern GladeXML * xml;
	GtkTreeIter iter;
	GtkTreeModel * model;
	GtkWidget * treeview = glade_xml_get_widget (xml, "treeview");
	GtkTreeSelection * selection;
	gsize bytes_written;
	GList  * rows;

	/* get the iterator from the current selection */
	selection = gtk_tree_view_get_selection ( GTK_TREE_VIEW (treeview));

	if ( gtk_tree_selection_count_selected_rows (selection) < 0 ) 
		return;

	rows = gtk_tree_selection_get_selected_rows (selection, &model);

	if ( rows == NULL )
		gtk_tree_model_get_iter_first (gtk_tree_view_get_model ( GTK_TREE_VIEW (treeview)), &iter);
	else
		gtk_tree_model_get_iter (model, &iter, rows->data);

	gtk_list_store_set (GTK_LIST_STORE (g->store), &iter,
			    TITLE_COLUMN, g_locale_to_utf8 (title, -1, NULL, &bytes_written, NULL),
			    TYPE_COLUMN, g_locale_to_utf8 (type, -1, NULL, &bytes_written, NULL),
			    STATUS_COLUMN, g_locale_to_utf8 (status, -1, NULL, &bytes_written, NULL),
			    LAST_UPDATE_COLUMN, g_locale_to_utf8 (last_update, -1, NULL, &bytes_written, NULL),
			    -1);
}

void
set_fields_value (gchar * title, gchar * text, int type, int status, gchar * date)
{
	extern GladeXML * xml;
	GtkTextIter start;
	GtkTextIter end;
	GtkWidget * entrysubject = glade_xml_get_widget (xml, "entrysubject");
	GtkWidget * textview = glade_xml_get_widget (xml, "textview");
	GtkWidget * optionmenutype = glade_xml_get_widget (xml, "optionmenutype");
	GtkWidget * optionmenustatus = glade_xml_get_widget (xml, "optionmenustatus");
	GtkWidget * ndate =  glade_xml_get_widget (xml, "labeldatevalue");
	gsize bytes_written;

	/* set the title field */
	gtk_entry_set_text ( GTK_ENTRY (entrysubject), g_locale_to_utf8(title, -1, NULL, &bytes_written, NULL));
	
	/* set the text field */
	GtkTextBuffer * buffer =  gtk_text_view_get_buffer ( GTK_TEXT_VIEW (textview) );	
	gtk_text_buffer_get_start_iter (buffer,&start);
	gtk_text_buffer_get_end_iter (buffer,&end);
	gtk_text_buffer_delete (buffer, &start, &end);
	if ( text == NULL )
		text = "";
	gtk_text_buffer_set_text (buffer, g_locale_to_utf8(text, -1, NULL, &bytes_written, NULL), strlen (text));

	scan_url_tag_from_textview (buffer, NULL);

	/* set the type option menu */
	if (type == NOTE)
		gtk_option_menu_set_history ( GTK_OPTION_MENU (optionmenutype), NOTE);
	else if (type == PERSONAL)
		gtk_option_menu_set_history ( GTK_OPTION_MENU (optionmenutype), PERSONAL);
	else if (type == PROBLEM)
		gtk_option_menu_set_history ( GTK_OPTION_MENU (optionmenutype), PROBLEM);
	else if (type == TRICK)
		gtk_option_menu_set_history ( GTK_OPTION_MENU (optionmenutype), TRICK);
	else if (type == BOOKMARK)
		gtk_option_menu_set_history ( GTK_OPTION_MENU (optionmenutype), BOOKMARK);

	/* set the status option menu */
	if (status == NONE)
		gtk_option_menu_set_history ( GTK_OPTION_MENU (optionmenustatus), NONE);
	else if (status == OPEN)
		gtk_option_menu_set_history ( GTK_OPTION_MENU (optionmenustatus), OPEN);
	else if (status == SOLVED)
		gtk_option_menu_set_history ( GTK_OPTION_MENU (optionmenustatus), SOLVED);


	/* set the date field */
	gtk_label_set_label ( GTK_LABEL (ndate), g_locale_to_utf8(date, -1, NULL, &bytes_written, NULL));
}

void
reset_option_menu (void)
{
	extern GladeXML * xml;
	GtkWidget * optionmenutype = glade_xml_get_widget (xml, "optionmenutype");
	GtkWidget * optionmenustatus = glade_xml_get_widget (xml, "optionmenustatus");

	gtk_option_menu_set_history ( GTK_OPTION_MENU (optionmenutype), NOTE);
	gtk_option_menu_set_history ( GTK_OPTION_MENU (optionmenustatus), NONE);
}

void
update_tree_with_result (gchar * query)
{
	extern global * g;
	note * n;
	int i;
	gchar * casefold_query = g_utf8_casefold (query, -1);

	clear_tree_view ();

	for ( i = 0 ; i < g->nl->notes->len ; i++ )
	{
		n = g_ptr_array_index (g->nl->notes, i);

		nb_note_set_text (n, nb_notelist_get_text_from_index (g->nl, i));

		if (  nb_note_matches_query (n, casefold_query) )
		{
			gchar * type;
			gchar * status;
			gchar * date;
			gchar * last_update;

			if (n->type == NOTE)
				type = (gchar *) _("Note");
			else if (n->type == PERSONAL)
				type = (gchar *) _("Personal");
			else if (n->type == PROBLEM)
				type = (gchar *) _("Problem");
			else if (n->type == TRICK)
				type = (gchar *) _("Trick");
			else if (n->type == BOOKMARK)
				type = (gchar *) _("Bookmark");

			if (n->status == NONE)
				status = (gchar *) _("None");
			else if (n->status == OPEN)
				status = (gchar *) _("Open");
			else if (n->status == SOLVED)
				status = (gchar *) _("Solved");

			/* convert the date and lastUpdate to String */
			date = g_malloc (128 * sizeof (gchar));
			last_update = g_malloc (128 * sizeof (gchar));
			g_date_strftime (date, 127, "%A %d %b", n->date);
			g_date_strftime (last_update, 127, "%A %d %b", n->lastUpdate);

			add_note_to_tree (n->title,
					type,
					status,
					date,
					last_update,
					n,
					FALSE);

			g_free (date);
			g_free (last_update);
		}

		nb_note_set_text (n, NULL);
	}
}

void
import_notes_from_file (void)
{
	extern global * g;
	extern GladeXML * xml;
	GtkWidget * dialog, * mainwindow;

	mainwindow = glade_xml_get_widget (xml, "mainwindow");

	dialog = gtk_file_chooser_dialog_new ((gchar *) _("Select file to import."), GTK_WINDOW (mainwindow),
					      GTK_FILE_CHOOSER_ACTION_OPEN,
					      GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
					      GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT,
					      NULL);

	if ( gtk_dialog_run ( GTK_DIALOG (dialog)) == GTK_RESPONSE_ACCEPT )
	{
		char * filename;

		filename = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (dialog));
		 if (filename != NULL)
			 nb_global_import_notelist (g, filename);

		 g_free (filename);
	}

	gtk_widget_destroy (dialog);
}

void export_notes_to_file (void)
{
	/* TODO: create the filechooser with libglade */
	extern GladeXML * xmlexport;
	GtkWidget * exportdialog, * hbox, * filechooser;

	/* create the export dialog */
	xmlexport = glade_xml_new (g_build_path ("/", GLADE_DIR, "exportdialog.glade", NULL), "exportdialog", NULL);
	glade_xml_signal_autoconnect(xmlexport);

	exportdialog = glade_xml_get_widget (xmlexport, "exportdialog");
	hbox = glade_xml_get_widget (xmlexport, "hbox1");
	filechooser = gtk_file_chooser_widget_new (GTK_FILE_CHOOSER_ACTION_SAVE);
	gtk_file_chooser_set_current_name ( GTK_FILE_CHOOSER (filechooser), DEFAULT_NOTES_FILE);
	gtk_widget_show (filechooser);

	gtk_box_pack_start ( GTK_BOX (hbox), filechooser, TRUE, TRUE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (filechooser), 6);

	g_object_set_data ( G_OBJECT (exportdialog), "filechooser", filechooser);
}

gchar *
get_current_date (void)
{
	GDate * date;
	gchar * cdate;

	date = g_date_new ();
	g_date_set_time (date, time (NULL));

	/* not very clean, should try to make it dynamic */
	cdate = g_malloc (128 * sizeof (gchar));
	g_date_strftime (cdate, 127, "%A %d %b", date);
	g_date_free (date);

	return cdate;
}

void
push_statusbar (gchar * text)
{
	extern GladeXML * xml;
	GtkWidget * statusbar;
	guint context_id;

	statusbar = glade_xml_get_widget (xml, "statusbar");
	context_id = gtk_statusbar_get_context_id ( GTK_STATUSBAR (statusbar), "main");

	gtk_statusbar_pop ( GTK_STATUSBAR (statusbar), context_id);
	gtk_statusbar_push ( GTK_STATUSBAR (statusbar), context_id, text);
}

void 
set_cursor_if_appropriate (GtkTextView * textview, gint x, gint y)
{
	GtkTextBuffer * buffer;
	GtkTextIter iter;
	GtkTextTagTable * table;
	GtkTextTag * tag;
	format_textview * format = g_object_get_data ( G_OBJECT (textview), "format");

	buffer = gtk_text_view_get_buffer (textview);

	gtk_text_view_get_iter_at_location (textview, &iter, x, y);

	table = gtk_text_buffer_get_tag_table (buffer);

	tag = gtk_text_tag_table_lookup (table, "links");

	if ( gtk_text_iter_has_tag (&iter, tag) && (format->current_cursor != GDK_HAND2))
		gdk_window_set_cursor (gtk_text_view_get_window (textview, GTK_TEXT_WINDOW_TEXT), format->hand_cursor);
	else
		gdk_window_set_cursor (gtk_text_view_get_window (textview, GTK_TEXT_WINDOW_TEXT), format->text_cursor);
}

void
store_interface_positions (void)
{
	extern GladeXML * xml;
	GConfClient * client;
	GtkWidget * mainwindow = glade_xml_get_widget (xml, "mainwindow");
	GtkWidget * vpan = glade_xml_get_widget (xml, "vpaned1");
	gint width, height, position;

	gdk_drawable_get_size ( GDK_DRAWABLE (mainwindow->window), &width, &height);
	position = gtk_paned_get_position ( GTK_PANED (vpan));

	client = gconf_client_get_default ();

	gconf_client_set_int (client, "/apps/notebook/ui/width", width, NULL);
	gconf_client_set_int (client, "/apps/notebook/ui/height", height, NULL);
	gconf_client_set_int (client, "/apps/notebook/ui/position", position, NULL);
}

void
load_interface_positions (void)
{
	extern GladeXML * xml;
	GConfClient * client;
	GtkWidget * mainwindow = glade_xml_get_widget (xml, "mainwindow");
	GtkWidget * vpan = glade_xml_get_widget (xml, "vpaned1");
	gint width = 0, height = 0, position = 0, actual_width, actual_height, actual_position;

	client = gconf_client_get_default ();

	width = gconf_client_get_int (client, "/apps/notebook/ui/width", NULL);
	height = gconf_client_get_int (client, "/apps/notebook/ui/height", NULL);
	position = gconf_client_get_int (client, "/apps/notebook/ui/position", NULL);

	gdk_drawable_get_size ( GDK_DRAWABLE (mainwindow->window), &actual_width, &actual_height);
	actual_position = gtk_paned_get_position ( GTK_PANED (vpan));

	if ( width != 0 )
		actual_width = width;

	if ( height != 0 ) 
		actual_height = height;

	if ( position != 0 )
		actual_position = position;

	gtk_window_resize ( GTK_WINDOW (mainwindow), actual_width, actual_height);
	gtk_paned_set_position ( GTK_PANED (vpan), actual_position);
}

void 	
create_about_dialog (void)
{
	const gchar * authors[] = { "Mathias André <mathias@openbrookes.org>", NULL};
	const gchar * version = "0.2";
	const gchar * comment = (gchar *) _("Notebook is a simple logbook editor.");
	const gchar * translator = (gchar *) _("Mathias André <mathias@openbrookes.org>");

	gtk_widget_show ( GTK_WIDGET (gnome_about_new (PACKAGE, VERSION,
			 "© 2004 Mathias André",
			 comment,
			 (const gchar **) authors,
			 NULL,
			 translator,
			 NULL)));
}
